import { supabase } from '@/config/supabase';
import {
  Contributor,
  Discount,
  AddContributorRequest,
  UpdateContributorRequest,
  CreateDiscountRequest,
  UpdateDiscountRequest
} from './types';

export const contributorsService = {
  // Contributors
  async addContributor(vendor_id: string, req: AddContributorRequest) {
    const { data, error } = await supabase.from('contributors').insert({
      vendor_id,
      name: req.name,
      email: req.email,
      role: req.role,
      permissions: req.permissions,
      is_active: true
    }).select().single();
    return { data, error };
  },

  async listContributors(vendor_id: string) {
    const { data, error } = await supabase.from('contributors').select('*').eq('vendor_id', vendor_id).eq('is_active', true);
    return { data, error };
  },

  async updateContributor(vendor_id: string, id: string, req: UpdateContributorRequest) {
    const { data, error } = await supabase.from('contributors').update(req).eq('id', id).eq('vendor_id', vendor_id).select().single();
    return { data, error };
  },

  async deleteContributor(vendor_id: string, id: string) {
    // Soft delete
    const { data, error } = await supabase.from('contributors').update({ is_active: false }).eq('id', id).eq('vendor_id', vendor_id).select().single();
    return { data, error };
  },

  // Discounts
  async createDiscount(vendor_id: string, req: CreateDiscountRequest) {
    const { data, error } = await supabase.from('discounts').insert({
      vendor_id,
      ...req,
      is_active: true
    }).select().single();
    return { data, error };
  },

  async listDiscounts(vendor_id: string, filters: any) {
    let query = supabase.from('discounts').select('*').eq('vendor_id', vendor_id);
    if (filters.type) query = query.eq('type', filters.type);
    if (filters.active !== undefined) query = query.eq('is_active', filters.active);
    if (filters.applies_to) query = query.eq('applies_to', filters.applies_to);
    if (filters.target_id) query = query.eq('target_id', filters.target_id);
    const { data, error } = await query.order('created_at', { ascending: false });
    return { data, error };
  },

  async applyDiscounts(vendor_id: string, customer_id: string, product_ids: string[]) {
    // Find best discounts for each product and customer
    const { data: discounts } = await supabase.from('discounts').select('*').eq('vendor_id', vendor_id).eq('is_active', true);
    const bestDiscounts: any[] = [];
    for (const pid of product_ids) {
      // Product-specific
      const productDiscounts = (discounts || []).filter(d => d.applies_to === 'product' && d.target_id === pid);
      // Customer-specific
      const customerDiscounts = (discounts || []).filter(d => d.applies_to === 'customer' && d.target_id === customer_id);
      // Order-wide
      const orderDiscounts = (discounts || []).filter(d => d.applies_to === 'order');
      // Pick best (highest value)
      const all = [...productDiscounts, ...customerDiscounts, ...orderDiscounts];
      if (all.length > 0) {
        const best = all.reduce((a, b) => (a.value > b.value ? a : b));
        bestDiscounts.push({ product_id: pid, discount: best });
      }
    }
    return { data: bestDiscounts };
  },

  async updateDiscount(vendor_id: string, id: string, req: UpdateDiscountRequest) {
    const { data, error } = await supabase.from('discounts').update(req).eq('id', id).eq('vendor_id', vendor_id).select().single();
    return { data, error };
  },

  async deleteDiscount(vendor_id: string, id: string) {
    // Soft delete
    const { data, error } = await supabase.from('discounts').update({ is_active: false }).eq('id', id).eq('vendor_id', vendor_id).select().single();
    return { data, error };
  }
}; 